function imgFeatures = extractFeatures(imgPath, fgSample, bgSample)
%EXTRACTFEATURES Extract color features from the image
%   For each pixel, the feature vector consists of both RGB and Lab color
%   channels weighted by geodesic destiance to both the foreground
%   scribbles and background scribbles. The size of the feature vector is
%   hence 12
% 
%   Input:
%       imgPath : path to the input image
%       fgSample: pixels of foreground scribbles
%       bgSample: pixels of background scribbles
% 
%   Output:
%       imgFeatures: A matrix of size height X width X 12 which represents
%       the feature vector for each pixel in the image
%
% (c) Moustafa Meshry, moustafa.meshry@alexu.edu.eg
%     Department of Compter and Systems Engineering, Alexandria University, Egypt.

    img_rgb = imread(imgPath);
    [L,A,B] = RGB2Lab(img_rgb);
    img_lab = zeros([size(img_rgb, 1) size(img_rgb, 2) 3]);
    img_lab(:,:,1) = L;
    img_lab(:,:,2) = A;
    img_lab(:,:,3) = B;

    % Normalize LAB
    L = L/100;
    A = (A+128)/256;
    B = (B+128)/256;
    
    img_lab_normalized = zeros([size(img_rgb, 1) size(img_rgb, 2) 3]);
    img_lab_normalized(:,:,1) = L;
    img_lab_normalized(:,:,2) = A;
    img_lab_normalized(:,:,3) = B;

    img_rgb_normalized = double(img_rgb)/255;

    [height, width, ~] = size(img_rgb);

    %% GeoDist from UNnormalized LAB/RGB channels
    
    fgDistTransform_lab = computeDistances(double(img_lab), fgSample');
    fgDistTransform_rgb = computeDistances(double(img_rgb), fgSample');
    fgDistWeighting_lab = 1 - fgDistTransform_lab / ((height + width - 2) * 255 * 3);
    fgDistWeighting_rgb = 1 - fgDistTransform_rgb / ((height + width - 2) * 255 * 3);
        
    bgDistTransform_lab = computeDistances(double(img_lab), bgSample');
    bgDistTransform_rgb = computeDistances(double(img_rgb), bgSample');
    bgDistWeighting_lab = 1 - bgDistTransform_lab / ((height + width - 2) * 255 * 3);
    bgDistWeighting_rgb = 1 - bgDistTransform_rgb / ((height + width - 2) * 255 * 3);

%%

    fgFeatureVector_lab = bsxfun(@times, img_lab_normalized, ...
        fgDistWeighting_lab);
    bgFeatureVector_lab = bsxfun(@times, img_lab_normalized, ...
        bgDistWeighting_lab);
    fgFeatureVector_rgb = bsxfun(@times, img_rgb_normalized, ...
        fgDistWeighting_rgb);
    bgFeatureVector_rgb = bsxfun(@times, img_rgb_normalized, ...
        bgDistWeighting_rgb);
    imgFeatures = cat(3, fgFeatureVector_lab, bgFeatureVector_lab, ...
        fgFeatureVector_rgb, bgFeatureVector_rgb);

end

